package MusicLandscape.util.formatters;

import MusicLandscape.entities.Track;
import MusicLandscape.util.MyFormatter;

/**
 * Format tracks to show all information in a tabular form.
 *
 * @author Jonas Altrock (ew20b126@technikum-wien.at)
 * @version 1
 * @since LabWork
 */
public class LongTrackFormatter implements MyFormatter<Track> {
    /**
     * The columns placeholders for String.format()
     */
    private final String cols = "%-21.21s ".repeat(3) + "%-4.4s";

    /**
     * the string representation of this formatter is
     * <kbd>"long format [Title Performer Writer Year (min:sec)]"</kbd>
     * (without quotes)
     *
     * @return the string representation
     */
    public String toString() {
        return "long format [Title Performer Writer Year (min:sec)]";
    }

    /**
     * Get the headers for the table as a single string.<br>
     * <p>
     * Contains the names for all columns separated by the correct number of
     * blanks.
     *
     * @return the header string.
     */
    @Override
    public String header() {
        return String.format(cols + " (%s:%s)", "Title", "Performer", "Writer", "Year", "min", "sec");
    }

    /**
     * A line of text to be used between header and data.
     *
     * @return the separator.
     */
    @Override
    public String topSeparator() {
        return "-".repeat(header().length());
    }

    /**
     * Creates a String representation for an object.
     *
     * @param track the object to be formatted
     * @return the formatted representing the object
     */
    @Override
    public String format(Track track) {
        return String.format(
                cols + " (%02d:%02d)",
                track.getTitle(),
                track.getPerformer(),
                track.getWriter(),
                track.getYear(),
                track.getDuration() / 60, track.getDuration() % 60
        );
    }
}
